using System;
using System.Net;
using System.Timers;
using System.Threading;
using System.Net.Sockets;
using System.Collections;
using System.Diagnostics;

using Team_Project.PersistencyManagers;

using Bertaccini.Utils;

namespace Team_Project.Utils
{
	/// <summary>
	/// Modulo adibito al controllo di raggiungibilit delle copie.
	/// Effettua gli "heart-beat".
	/// </summary>
	/// <remarks>L'istanza globale di questa classe  disponibile dalle
	/// costanti globali con il nome "AliveMonitor"</remarks>
	[TeamProjectInit(true,"AliveMonitor")]
	public class AliveMonitor:MarshalByRefObject, IDisposable
	{
		/// <summary>
		/// Collezione degli end points remoti.
		/// </summary>
		protected ArrayList remoteEndPoints = new ArrayList();
		/// <summary>
		/// Nomi delle copie da controllare.
		/// </summary>
		protected ArrayList Names = new ArrayList();
		/// <summary>
		/// Array contenente il time-stamp dell'ultimo messaggio ricevuto da una
		/// copia.
		/// </summary>
		protected ArrayList LastAlive = new ArrayList();
		/// <summary>
		/// Thread sender dell'heart-beat
		/// </summary>
		protected Thread S;
		/// <summary>
		/// Thread listener degli heart-beat
		/// </summary>
		protected Thread L;
		/// <summary>
		/// Timer utilizzato per il controllo degli heart-beat ricevuti
		/// </summary>
		protected System.Timers.Timer tickl;
		/// <summary>
		/// Controlla se i loop di send e receive devono continuare.
		/// Quando  impostato a false i thread terminano al prossimo tick
		/// </summary>
		protected bool continuE = true;
		/// <summary>
		/// Tempo in milli-secondi ogni quanto viene inviato un heart-beat
		/// </summary>
		public static int AliveCycle = 10000;
		/// <summary>
		/// Moltiplicatore di tolleranza. Il valore di questo campo controlla
		/// ogni quanto viene effettuato il check degli heart-beat ricevuti
		/// </summary>
		public static double TolleranceMultiplier = 3.5;
		/// <summary>
		/// Oggetto di sincronizzazione dei thread
		/// </summary>
		protected Mutex2 sync = new Mutex2(0,"AliveMonitorMutex");

		/// <summary>
		/// Inizializza un'istanza di Alive Monitor. Richiesto all'avvio del servizio
		/// dall'engine di team-project
		/// </summary>
		/// <returns>Un'alive monitor configurato</returns>
		public static AliveMonitor ComponentInit()
		{
			AliveMonitor me = new AliveMonitor();
			CopiesTreeManager ctm = (CopiesTreeManager)Globals.Instance.Data["TreeManager"];
			string par = ctm.GetParent();
			if (par != null && par != "")
				me.AddEndPoint(new IPEndPoint(MyDNS.ResolveIP(par),
					MyDNS.ListenerPort(par)),par);
			foreach (string c in ctm.GetChildren())
			{
				me.AddEndPoint(new IPEndPoint(MyDNS.ResolveIP(c),
					MyDNS.ListenerPort(c)),c);
			}
			ctm.CopyRemoved += new CopyEventDelegate(me.ctm_CopyRemoved);
			ctm.CopyAdded +=new CopyEventDelegate(me.ctm_CopyAdded);
			return me;
		}

		/// <summary>
		/// Crea un'istanza di aliveMonitor. Inizializza e fa partire i thread
		/// e il timer necessari
		/// </summary>
		public AliveMonitor()
		{
			S = new Thread(new ThreadStart(Sender));
			L = new Thread(new ThreadStart(Listener));
			tickl = new System.Timers.Timer(AliveCycle * TolleranceMultiplier);
			tickl.Elapsed +=new ElapsedEventHandler(tickl_Elapsed);
			S.Start();
			L.Start();
			tickl.Start();
		}

		#region Threads code
		private void Sender()
		{
			while(continuE)
			{
				lock(this)
				{
					foreach(IPEndPoint ep in remoteEndPoints)
					{
						UdpClient conn = new UdpClient();
						conn.Send(System.Text.Encoding.ASCII.GetBytes(Globals.Instance.LocalCopyName),
							Globals.Instance.LocalCopyName.Length,ep);
						Trace.WriteLine("Sent udp packet to: " + ep.ToString());
					}
				}
				Thread.Sleep(AliveCycle);
			}
		}

		private void Listener()
		{
			try
			{
				EndPoint ep = (EndPoint)new IPEndPoint(IPAddress.Any,0);
				Socket skt = new Socket(ep.AddressFamily,SocketType.Dgram,ProtocolType.Udp);
				byte[] data = new byte[256];
				EndPoint local = new IPEndPoint(IPAddress.Any,
					MyDNS.ListenerPort(Globals.Instance.LocalCopyName));
				skt.Bind(local);

				ReceiveDelegate RDel = new ReceiveDelegate(skt.ReceiveFrom);
				while(continuE)
				{
					Pair args = new Pair(RDel,data);
					RDel.BeginInvoke(data,ref ep,new AsyncCallback(MessageReceived),args);
					sync.Wait();
					Trace.WriteLine("AliveListener: Signaled. Continue="+continuE.ToString());
				}
			}
			catch(ThreadAbortException)
			{
				Trace.WriteLine("Listener thread aborted. Continue="+continuE.ToString());
			}
		}
		private delegate int ReceiveDelegate(byte[] data,ref EndPoint ep);

		private void MessageReceived(IAsyncResult ar)
		{
			try
			{
				EndPoint dcEp = new IPEndPoint(IPAddress.Any,0);
				ReceiveDelegate RDel = (ReceiveDelegate)((Pair)ar.AsyncState).First;
				int count  = RDel.EndInvoke(ref dcEp,ar);
				byte[] data = (byte[])((Pair)ar.AsyncState).Second;
				string val = System.Text.Encoding.ASCII.GetString(data,0,count);
				Trace.WriteLine("Received packet from: " + dcEp.ToString() + " " +
					val);
			
				lock(this)
				{
					int p = Names.IndexOf(val);
					if(p >= 0)
						LastAlive[p] = DateTime.Now;
					else
						Trace.WriteLine("Copy not found");
				}

			}
			catch(Exception e)
			{
				Trace.WriteLine(e.Message);
			}
			finally
			{
				Trace.WriteLine("AliveMonitor Synck signaled by MessageReceived");
				sync.Signal();
			}
		}

		#endregion

		/// <summary>
		/// Aggiunge un end-point di cui controllare la raggiungibilit
		/// </summary>
		/// <param name="ep">End-point della copia da aggiungere</param>
		/// <param name="name">Nome della copia da aggiungere</param>
		public void AddEndPoint(IPEndPoint ep,string name)
		{
			lock(this){
				remoteEndPoints.Add(ep);
				LastAlive.Add(DateTime.Now);
				Names.Add(name);
			}
		}

		/// <summary>
		/// Trova l'indice all'interno degli array della copia corrispondente
		/// all'end-point in parametro
		/// </summary>
		/// <param name="ep">End-point di cui ottenere l'indice</param>
		/// <returns>L'indice all'interno degli array dell'end-point specificato
		/// oppure -1 se esso non viene trovato</returns>
		protected int Find(IPEndPoint ep)
		{
			for(int i = 0; i < remoteEndPoints.Count; i++)
			{
				IPEndPoint ep2 = (IPEndPoint)remoteEndPoints[i];
				if(ep2.Address.Equals(ep.Address) && ep2.Port == ep.Port)
				{
					return i;
				}
			}
			return -1;
		}

		/// <summary>
		/// Rimuove un end-point e gli elementi corrispondenti negli altri array.
		/// La copia corrispondente non sar pi monitorata n le saranno inviati
		/// ulteriori messaggi
		/// </summary>
		/// <param name="ep">End-point corrispondente alla copia da eliminare</param>
		public void RemoveEndPoint(IPEndPoint ep)
		{
			lock(this)
			{
				int p = Find(ep);
				if(p >= 0)
				{
					RemoveIndex(p);
				}
			}
		}

		/// <summary>
		/// Rimuove da tutti gli array l'indice in parametro
		/// </summary>
		/// <param name="i">Indice dell'array da eliminare</param>
		protected void RemoveIndex(int i)
		{
			remoteEndPoints.RemoveAt(i);
			LastAlive.RemoveAt(i);
			Names.RemoveAt(i);
		}

		/// <summary>
		/// Rimuove una copia dalla lista dei monitorati a partire dal suo nome
		/// </summary>
		/// <param name="name">Nome della copia da eliminare</param>
		public void RemoveByName(string name)
		{
			Trace.WriteLine("Removing " + name);
			lock(this)
			{
				int p = Names.IndexOf(name);
				if(p >= 0)
				{
					RemoveIndex(p);
				}
			}
		}

		private void tickl_Elapsed(object sender, ElapsedEventArgs e)
		{
			ArrayList deadNames = new ArrayList(2);
			lock(this)
			{
				for(int i = 0; i < LastAlive.Count;i++)
				{
					TimeSpan ts = e.SignalTime-(DateTime)LastAlive[i];
					double elapsed = ts.TotalMilliseconds;
					if(elapsed > AliveCycle)
					{
						deadNames.Add(Names[i]);
					}
				}
			}
			foreach(string cp in deadNames)
			{
				DeadCopyFound(cp);
			}
		}

		/// <summary>
		/// Funzione richiamata quando viene trovata una copia irraggiungibile
		/// </summary>
		/// <remarks>Notifica il CopyManager che una copia non risulta pi
		/// disponibile</remarks>
		/// <param name="n">Nome della copia trovata</param>
		public virtual void DeadCopyFound(string n)
		{
			Trace.WriteLine("AliveMonitor: unreachable copy found: " + n);
			CopiesTreeManager ctm = (CopiesTreeManager)Globals.Instance.Data["TreeManager"];
			ctm.RemoveCopy(n);
		}

		#region IDisposable Members
		/// <summary>
		/// Rilascia le risorse occupate fermando anche i thread di send e listen
		/// </summary>
		public void Dispose()
		{
			continuE = false;
			tickl.Stop();
			if(!sync.Avaiable)
			{
				Trace.WriteLine("AliveMonitor Synck signaled by Dispose");
				sync.Signal();
			}
			try
			{
				L.Abort();
			}
			catch(Exception e)
			{Trace.WriteLine("Abort Exception on the main thread:"+e.Message);}
		}

		#endregion

		private void ctm_CopyRemoved(object source, TPCopyEventArgs args)
		{
			this.RemoveByName(args.CopyName);
		}

		private void ctm_CopyAdded(object source, TPCopyEventArgs args)
		{
			this.AddEndPoint(new IPEndPoint(MyDNS.ResolveIP(args.CopyName),
				MyDNS.ListenerPort(args.CopyName)),args.CopyName);
		}
	}
}